/*
 +----------------------------------------------------------------------+
 | This file is part of the Ayyi project. https://www.ayyi.org          |
 | copyright (C) 2025-2025 Tim Orford <tim@orford.org>                  |
 +----------------------------------------------------------------------+
 | This program is free software; you can redistribute it and/or modify |
 | it under the terms of the GNU General Public License version 3       |
 | as published by the Free Software Foundation.                        |
 +----------------------------------------------------------------------+
 |
 */

#include "config.h"
#include "agl/actor.h"
#include "test/unit.h"

extern AGl _agl;

extern bool agl_actor__is_onscreen (AGlActor*);

AGlScene scene = {
	.type = 4,
	.actor = {
		.name = "root",
		.region = { .x2 = 100., .y2 = 10. }
	}
};

AGlActor container = {
	.name = "container",
	.parent = (AGlActor*)&scene,
	.root = &scene,
	.region = { .x2 = 100., .y2 = 10. },
	.scrollable = { .x2 = 100, .y2 = 10 },
};


void
test_offset ()
{
	START_TEST;

	AGlActor* actor = agl_actor__new(AGlActor,
		.region = { .x2 = 100, .y2 = 10 },
		.parent = &container,
		.root = &scene,
	);

	extern AGliPt agl_actor__find_offset (AGlActor* a);

	AGliPt offset = agl_actor__find_offset(actor);
	assert(offset.x == 0, "%i", offset.x);

	actor->region.x1 = 10.;
	offset = agl_actor__find_offset(actor);
	assert(offset.x == 10, "%i", offset.x);

	actor->region.x1 = -10.;
	offset = agl_actor__find_offset(actor);
	assert(offset.x == -10, "%i", offset.x);

	// container is offset

	actor->region.x1 = 0.;
	container.region.x1 = -10.;
	offset = agl_actor__find_offset(actor);
	assert(offset.x == -10, "%i", offset.x);

	actor->region.x1 = -10.;
	container.region.x1 = -10.;
	offset = agl_actor__find_offset(actor);
	assert(offset.x == -20, "%i", offset.x);

	// container is scrolled

	actor->region.x1 = 0.;
	container.region.x1 = 0.;
	container.scrollable.x1 = -10.;
	offset = agl_actor__find_offset(actor);
	assert(offset.x == -10, "%i", offset.x);

	actor->region.x1 = -10.;
	container.region.x1 = 0.;
	container.scrollable.x1 = -10.;
	offset = agl_actor__find_offset(actor);
	assert(offset.x == -20, "%i", offset.x);

	// actor and container are scrolled

	actor->region.x1 = -10.;
	actor->scrollable.x1 = -10.;
	container.region.x1 = 0.;
	container.scrollable.x1 = -10.;
	offset = agl_actor__find_offset(actor);
	assert(offset.x == -30, "%i", offset.x);

	agl_actor__free(actor);

	FINISH_TEST;
}


void
test_is_onscreen ()
{
	START_TEST;

	AGlActor* actor = agl_actor__new(AGlActor,
		.region = { .x2 = 100, .y2 = 10 },
		.parent = &container,
		.root = &scene,
	);

	container.region = (AGlfRegion){ .x2 = 100., .y2 = 10. };
	container.scrollable = (AGliRegion){ .x2 = 100, .y2 = 10 };

	assert(agl_actor__is_onscreen(actor), "onscreen");

	actor->region = (AGlfRegion){ .x1 = 50., .x2 = 150. };
	assert(agl_actor__is_onscreen(actor), "onscreen");

	actor->region = (AGlfRegion){ .x1 = 101., .x2 = 200. };
	assert(!agl_actor__is_onscreen(actor), "not onscreen");

	// actor scrollable

	actor->region = (AGlfRegion){ .x1 = 0., .x2 = 100. };
	actor->scrollable = (AGliRegion){ .x1 = -100, .x2 = 100 };
	assert(agl_actor__is_onscreen(actor), "onscreen");

	actor->region = (AGlfRegion){ .x1 = 0., .x2 = 100. };
	actor->scrollable = (AGliRegion){ .x1 = 0, .x2 = 200 };
	assert(agl_actor__is_onscreen(actor), "onscreen");

	// parent scrollable

	actor->region = (AGlfRegion){ .x1 = 0., .x2 = 50. };
	actor->scrollable = (AGliRegion){ 0 };
	container.scrollable = (AGliRegion){ .x1 = -100, .x2 = 200 };
	assert(!agl_actor__is_onscreen(actor), "scrolled offscreen to left");

	actor->region = (AGlfRegion){ .x1 = 50., .x2 = 150. };
	assert(agl_actor__is_onscreen(actor), "partially scrolled offscreen");

	actor->region = (AGlfRegion){ .x1 = 150., .x2 = 200. };
	assert(agl_actor__is_onscreen(actor), "partially scrolled offscreen");

	actor->region = (AGlfRegion){ .x1 = 250., .x2 = 300. };
	assert(!agl_actor__is_onscreen(actor), "scrolled offscreen to right");

	agl_actor__free(actor);

	FINISH_TEST;
}


void
test_calc_viewport ()
{
	START_TEST;

	g_autoptr(AGlActor) actor = agl_actor__new(AGlActor,
		.region = { .x2 = 100, .y2 = 10 },
		.parent = &container,
		.root = &scene,
	);

	container.region = (AGlfRegion){ .x2 = 100., .y2 = 10. };
	container.scrollable = (AGliRegion){ .x2 = 100, .y2 = 10 };

	char* label;
	AGlfRegion region;
	AGlActor* root = (AGlActor*)&scene;

	#define assert_region_equal(region, expected) \
		assert(region.x1 == expected.x1, "%s: x1: got %.0f, expected %.0f", label, region.x1, expected.x1); \
		assert(region.x2 == expected.x2, "%s: x2: got %.0f, expected %.0f", label, region.x2, expected.x2); \
		assert(region.y1 == expected.y1, "%s: y1: got %.0f, expected %.0f", label, region.y1, expected.y1); \
		assert(region.y2 == expected.y2, "%s: y2: got %.0f, expected %.0f", label, region.y2, expected.y2);

	label = "no translation";
	agl_actor__calc_visible (actor, &region);
	AGlfRegion expected = {0, 0, 100, 10};
	assert_region_equal (region, expected);

	label = "simple offset (cropped on rhs)";
	actor->region.x1 = 12;
	agl_actor__calc_visible (actor, &region);
	expected = (AGlfRegion){0, 0, 88, 10};
	assert_region_equal (region, expected);

	label = "simple offset (not cropped on rhs)";
	actor->region.x1 = 12;
	actor->region.x2 = 60;
	agl_actor__calc_visible (actor, &region);
	expected = (AGlfRegion){0, 0, 48, 10};
	assert_region_equal (region, expected);

	label = "negative offset (not cropped on rhs)";
	actor->region.x1 = -12;
	actor->region.x2 = 60;
	agl_actor__calc_visible (actor, &region);
	expected = (AGlfRegion){12, 0, 72, 10};
	assert_region_equal (region, expected);

	label = "negative offset (cropped on rhs)";
	actor->region.x1 = -12;
	actor->region.x2 = 200;
	agl_actor__calc_visible (actor, &region);
	expected = (AGlfRegion){12, 0, 112, 10};
	assert_region_equal (region, expected);

	label = "scroll offset";
	actor->region.x1 = 0;
	actor->region.x2 = 100;
	actor->scrollable.x1 = -11;
	agl_actor__calc_visible (actor, &region);
	expected = (AGlfRegion){11, 0, 100, 10};
	assert_region_equal (region, expected);

	label = "scroll offset, positive"; // the result here is not correct because this scenario is not supported
	actor->region.x1 = 0;
	actor->region.x2 = 100;
	actor->scrollable.x1 = 11;
	agl_actor__calc_visible (actor, &region);
	expected = (AGlfRegion){0, 0, 89, 10};
	assert_region_equal (region, expected);

	label = "scroll offset, rhs cropped";
	actor->region.x1 = 0;
	actor->region.x2 = 100;
	actor->scrollable.x1 = -11;
	actor->scrollable.x2 = 200;
	agl_actor__calc_visible (actor, &region);
	expected = (AGlfRegion){11, 0, 111, 10};
	assert_region_equal (region, expected);

	label = "offset and scrollable";
	actor->region.x1 = 10;
	actor->region.x2 = 100;
	actor->scrollable.x1 = -11;
	actor->scrollable.x2 = 200;
	agl_actor__calc_visible (actor, &region);
#ifdef CROP_TO_ALL
	expected = (AGlfRegion){11, 0, 101, 10};
#else
	// currently our calculation doesn't crop to intermediate actors
	expected = (AGlfRegion){1, 0, 101, 10};
#endif
	assert_region_equal (region, expected);

	label = "parent offset";
	actor->region.x1 = 0;
	actor->region.x2 = 100;
	actor->scrollable.x1 = 0;
	actor->scrollable.x2 = 100;
	container.region.x1 = 9;
	agl_actor__calc_visible (actor, &region);
	expected = (AGlfRegion){0, 0, 91, 10};
	assert_region_equal (region, expected);

	label = "parent negative offset";
	actor->region.x1 = 0;
	actor->region.x2 = 100;
	actor->scrollable.x1 = 0;
	actor->scrollable.x2 = 100;
	container.region.x1 = -9;
	agl_actor__calc_visible (actor, &region);
	expected = (AGlfRegion){9, 0, 100, 10};
	assert_region_equal (region, expected);

	label = "parent offset, cropped rhs";
	actor->region.x1 = 0;
	actor->region.x2 = 100;
	actor->scrollable.x1 = 0;
	actor->scrollable.x2 = 100;
	container.region.x1 = 9;
	container.region.x2 = 90;
	agl_actor__calc_visible (actor, &region);
#if CROP_TO_ALL
	expected = (AGlfRegion){0, 0, 81, 10};
#else
	expected = (AGlfRegion){0, 0, 91, 10};
#endif
	assert_region_equal (region, expected);

	label = "parent scrollable";
	actor->region.x1 = 0;
	actor->region.x2 = 100;
	actor->scrollable.x1 = 0;
	actor->scrollable.x2 = 100;
	container.region.x1 = 0;
	container.region.x2 = 100;
	container.scrollable.x1 = -20;
	container.scrollable.x2 = 100;
	agl_actor__calc_visible (actor, &region);
	expected = (AGlfRegion){20, 0, 100, 10};
	assert_region_equal (region, expected);

	label = "parent scrollable, scrollregion unset";
	actor->region.x1 = 0;
	actor->region.x2 = 100;
	actor->scrollable.x1 = 0;
	actor->scrollable.x2 = 0;
	container.region.x1 = 0;
	container.region.x2 = 100;
	container.scrollable.x1 = -20;
	container.scrollable.x2 = 100;
	agl_actor__calc_visible (actor, &region);
	expected = (AGlfRegion){20, 0, 100, 10};
	assert_region_equal (region, expected);

	label = "parent scrollable, offset";
	actor->region.x1 = 950;
	actor->region.x2 = 1050;
	actor->scrollable.x1 = 0;
	actor->scrollable.x2 = 100;
	container.region.x1 = 0;
	container.region.x2 = 100;
	container.scrollable.x1 = -1000;
	container.scrollable.x2 = 1000;
	agl_actor__calc_visible (actor, &region);
	expected = (AGlfRegion){50, 0, 100, 10};
	assert_region_equal (region, expected);

	label = "grand-parent scrollable";
	actor->region.x1 = 0;
	actor->region.x2 = 100;
	actor->scrollable.x1 = 0;
	actor->scrollable.x2 = 100;
	container.region.x1 = 0;
	container.region.x2 = 100;
	container.scrollable.x1 = 0;
	container.scrollable.x2 = 100;
	root->scrollable.x1 = -25;
	root->scrollable.x2 = 130;
	agl_actor__calc_visible (actor, &region);
	expected = (AGlfRegion){25, 0, 100, 10};
	assert_region_equal (region, expected);

	label = "cumulative scrollable";
	actor->region.x1 = 0;
	actor->region.x2 = 100;
	actor->scrollable.x1 = 0;
	actor->scrollable.x2 = 100;
	container.region.x1 = 0;
	container.region.x2 = 100;
	container.scrollable.x1 = -7;
	container.scrollable.x2 = 100;
	root->scrollable.x1 = -25;
	root->scrollable.x2 = 130;
	agl_actor__calc_visible (actor, &region);
	expected = (AGlfRegion){32, 0, 100, 10};
	assert_region_equal (region, expected);

	label = "large scrollable, spans viewport";
	actor->region.x1 = 7143;
	actor->region.x2 = 14286;
	actor->scrollable.x1 = 0;
	actor->scrollable.x2 = 0;
	container.region.x1 = 0;
	container.region.x2 = 50000;
	container.scrollable.x1 = 0;
	container.scrollable.x2 = 0;
	container.scrollable.y2 = 0;
	root->scrollable.x1 = -7152;
	root->scrollable.x2 = 42848;
	agl_actor__calc_visible (actor, &region);
	expected = (AGlfRegion){9, 0, 109, 10};
	assert_region_equal (region, expected);

	label = "large scrollable, starts in middle of viewport";
	actor->region.x1 = 7000;
	actor->region.x2 = 14000;
	actor->scrollable.x1 = 0;
	actor->scrollable.x2 = 0;
	container.region.x1 = 0;
	container.region.x2 = 50000;
	container.scrollable.x1 = 0;
	container.scrollable.x2 = 0;
	root->scrollable.x1 = -6990;
	root->scrollable.x2 = 43010;
	agl_actor__calc_visible (actor, &region);
	expected = (AGlfRegion){0, 0, 90, 10};
	assert_region_equal (region, expected);

	FINISH_TEST;
}
