/*
 +----------------------------------------------------------------------+
 | This file is part of the Ayyi project. https://www.ayyi.org          |
 | copyright (C) 2025-2025 Tim Orford <tim@orford.org>                  |
 +----------------------------------------------------------------------+
 | This program is free software; you can redistribute it and/or modify |
 | it under the terms of the GNU General Public License version 3       |
 | as published by the Free Software Foundation.                        |
 +----------------------------------------------------------------------+
 |
 */

#include "config.h"
#include <glib.h>
#include "debug/debug.h"
#include "agl/actor.h"
#include "transition_observable.h"


typedef struct {
   AGlObservableFn fn;
   gpointer        user;
} Subscription;


AGlTransitionObservable*
transition_observable_new ()
{
	AGlTransitionObservable* observable = AGL_NEW(AGlTransitionObservable,
		.observable = {
			.max.i = INT_MAX
		},
		.animatable = {
			.type = WF_INT,
		}
	);
	observable->animatable.val.i = &observable->observable.value.u;

	void transition_observable_on_scroll (AGlObservable* o, AGlVal value, gpointer _actor)
	{
		// sync the animatable after a non-animated change

		AGlTransitionObservable* tobs = (AGlTransitionObservable*)o;
		if (!tobs->animation) {
			tobs->animatable.target_val.b = value.b;
		}
	}
	agl_observable_subscribe((AGlObservable*)observable, transition_observable_on_scroll, NULL);

	return observable;
}


bool
transition_observable_set_int (AGlTransitionObservable* observable, int value)
{
	value = CLAMP(value, observable->observable.min.i, observable->observable.max.i);
	if (value == observable->observable.value.i) return false;

	observable->animatable.target_val.i = value;
	observable->animatable.start_val.i = observable->observable.value.i;

	void transition_observable_on_frame (WfAnimation* animation, int time)
	{
		AGlTransitionObservable* observable = animation->user_data;

		for(GList* l = observable->observable.subscriptions;l;l=l->next){
			Subscription* subscription = l->data;
			subscription->fn((AGlObservable*)observable, observable->observable.value, subscription->user);
		}
	}

	void transition_observable_on_animation_finished (WfAnimation* animation, gpointer user_data)
	{
		AGlTransitionObservable* observable = animation->user_data;
		observable->animation = NULL;
	}

	if (observable->animation) {
		wf_transition_update(observable->animation);
	} else {
		observable->animation = wf_animation_new(transition_observable_on_animation_finished, observable);
		observable->animation->on_frame = transition_observable_on_frame;

		wf_transition_add_member(observable->animation, g_list_prepend(NULL, &observable->animatable));

		wf_animation_start(observable->animation);
	}

	return true;
}
