/*
 +----------------------------------------------------------------------+
 | This file is part of the Ayyi project. https://www.ayyi.org          |
 | copyright (C) 2021-2025 Tim Orford <tim@orford.org>                  |
 +----------------------------------------------------------------------+
 | This program is free software; you can redistribute it and/or modify |
 | it under the terms of the GNU General Public License version 3       |
 | as published by the Free Software Foundation.                        |
 +----------------------------------------------------------------------+
 | behaviours/scrollable_h.c
 |
 | Manages horizontal scrolling of an actor.
 | Adds a sibling scrollbar widget and an observable for the scroll
 | position.
 | Handles mouse and keyboard events for scrolling.
 | Does not set the size of the actor->scrollable region, but does
 | consume it, and will update the position when scrolling.
 |
 */

#include "config.h"
#include "debug/debug.h"
#undef USE_GTK
#include <gdk/gdkkeysyms.h>
#include "actors/scrollbar.h"
#include "scrollable_h.h"

#define SCROLL_INCREMENT 10
#define max_scroll_offset (MAX(0, agl_actor__scrollable_width(actor) - (int)agl_actor__width(actor)))

static void scrollable_free      (AGlBehaviour*);
static void scrollable_init      (AGlBehaviour*, AGlActor*);
static void scrollable_layout    (AGlBehaviour*, AGlActor*);
static bool scrollable_event     (AGlBehaviour*, AGlActor*, GdkEvent*);
static void scrollable_on_scroll (AGlObservable*, AGlVal, gpointer);

static AGlActor* scrollable_find_scrollbar (AGlActor*);

static AGlBehaviourClass klass = {
	.new = scrollable_h,
	.free = scrollable_free,
	.init = scrollable_init,
	.layout = scrollable_layout,
	.event = scrollable_event
};


AGlBehaviourClass*
scrollable_h_get_class ()
{
	return &klass;
}


AGlBehaviour*
scrollable_h ()
{
	HScrollableBehaviour* a = AGL_NEW (HScrollableBehaviour,
		.behaviour = {
			.klass = &klass,
		},
		.scroll = transition_observable_new ()
	);

	return (AGlBehaviour*)a;
}


static void
scrollable_free (AGlBehaviour* behaviour)
{
	HScrollableBehaviour* scrollable = (HScrollableBehaviour*)behaviour;

	agl_observable_free((AGlObservable*)scrollable->scroll);

	g_free(scrollable);
}


void
scrollable_init (AGlBehaviour* behaviour, AGlActor* actor)
{
	HScrollableBehaviour* scrollable = (HScrollableBehaviour*)behaviour;

	AGlActor* scrollbar = scrollable_find_scrollbar (actor);
	if (scrollbar) return;

	scrollbar = scrollbar_view (actor, AGL_ORIENTATION_HORIZONTAL, (AGlObservable*)scrollable->scroll, NULL, 1);
	scrollbar->z = actor->z + 10;
	agl_actor__add_child (actor->parent, scrollbar);
	agl_observable_subscribe((AGlObservable*)scrollable->scroll, scrollable_on_scroll, actor);
}


static AGlActor*
scrollable_find_scrollbar (AGlActor* actor)
{
	AGlActor* parent = actor->parent;
	if (!parent) return NULL;

	for (GList* l = parent->children;l;l=l->next) {
		if (((AGlActor*)l->data)->class == scrollbar_view_get_class()) return l->data;
	}

	return NULL;
}


static void
scrollable_layout (AGlBehaviour* behaviour, AGlActor* actor)
{
	HScrollableBehaviour* scrollable = (HScrollableBehaviour*)behaviour;

	((AGlObservable*)scrollable->scroll)->max.i = max_scroll_offset;
	((AGlObservable*)scrollable->scroll)->value.i = -actor->scrollable.x1;
}


static bool
scrollable_event (AGlBehaviour* behaviour, AGlActor* actor, GdkEvent* event)
{
	HScrollableBehaviour* scrollable = (HScrollableBehaviour*)behaviour;

	bool scrollable_set_position (HScrollableBehaviour* scrollable, int scroll_offset)
	{
		g_return_val_if_fail (max_scroll_offset > -1, AGL_HANDLED);

		return transition_observable_set_int (scrollable->scroll, (scroll_offset / SCROLL_INCREMENT) * SCROLL_INCREMENT);
	}

	bool inc ()
	{
		return scrollable_set_position (scrollable, scrollable->scroll->animatable.target_val.i + SCROLL_INCREMENT + 1);
	}

	bool dec ()
	{
		return scrollable_set_position (scrollable, scrollable->scroll->animatable.target_val.i - SCROLL_INCREMENT);
	}

	switch (event->type) {
		case GDK_BUTTON_PRESS:
			switch (event->button.button) {
				case 4:
					return dec();
				case 5:
					if (agl_actor__scrollable_width (actor) > agl_actor__width(actor)) {
						return inc();
					}
			}
			break;
		case GDK_SCROLL:
			return (((GdkEventScroll*)event)->direction == GDK_SCROLL_UP)
				? inc()
				: dec();

		case GDK_KEY_PRESS:
			;GdkEventKey* e = (GdkEventKey*)event;
			switch (e->keyval) {
				case GDK_KEY_Left:
					return dec();
				case GDK_KEY_Right:
					return inc();
				default:
					break;
			}
			break;
		default:
			break;
	}
	return AGL_NOT_HANDLED;
}


static void
scrollable_on_scroll (AGlObservable* o, AGlVal value, gpointer _actor)
{
	AGlActor* actor = _actor;

	int scrollable_width = agl_actor__scrollable_width(actor);

	actor->scrollable.x1 = - value.i;
	actor->scrollable.x2 = actor->scrollable.x1 + scrollable_width;

	agl_scene_queue_draw ((AGlScene*)actor->root);
}
