/*
  copyright (C) 2009 Tim Orford <tim@orford.org>

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License version 3
  as published by the Free Software Foundation.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
*/
#define __window_c__
#include <stdlib.h>
#include <stdint.h>
#include <string.h>
#include <gtk/gtk.h>
#include <gdk/gdkkeysyms.h>
#include "fst.h"

#include "jackvst.h"
#include "ayyi_utils.h"
typedef struct _plug plug;
typedef struct _plugin_window plugin_window;
#include "window.h"

#define HANDLED TRUE
#define NOT_HANDLED FALSE
#define NON_HOMOGENOUS 0

extern GtkTargetEntry sm_dnd_file_drag_types[];
extern JackVST*       jvst;
extern void           easyfst_quit();

static void       plugin_window_drag_dataget(GtkWidget*, GdkDragContext*, GtkSelectionData*, guint, guint, gpointer);
static gboolean   plugin_treeview_on_button_press(GtkWidget*, GdkEventButton*, gpointer);
static gboolean   on_popupmenu(GtkWidget*, gpointer);
static void       show_popupmenu(GtkWidget*, GdkEventButton*, gpointer);
static GtkWidget* context_menu_init(GtkTreeView* treeview);
static void       run_selected_plugin(GtkWidget*, GtkTreeView*);
static gboolean   destroy_handler(GtkWidget*, GdkEventAny*, gpointer);

int           debug;
guint         click_timer = 0;
GtkTreeStore* plugin_treestore;
GtkTreeModel* sort_model = NULL; 
GHashTable*   plugin_icons;
GtkWidget*    statusbar;
GtkWidget*    menu;

struct _plugins
{
	GHashTable* map;
} plugins;

typedef struct _plugin {
	FSTInfo* finfo;
	char*    dir;
} Plugin;

void
plugins_init()
{
	PF;
	pluglist = NULL;

	plugin_treestore = GTK_TREE_STORE(plugins_create_model());

	plugins_load_icons();

	const char* vst_path = g_getenv("VST_PATH");
	if(!vst_path) vst_path = "/usr/share/vst/";
	vst_discover_from_path((char*)vst_path);
}


void
plugins_free()
{
	GList* l= pluglist;
	for(;l;l=l->next){
		Plugin* plugin = l->data;
		free(plugin->finfo);
		free(plugin->dir);
		free(plugin);
	}
	g_list_free(pluglist);
	pluglist = NULL;
}


void
vst_init()
{
  PF;
}


void
vst_get_info()
{
}


#if 0
void
vst_plugin_add(int shm_slot, plugin_shared* shared)
{
  plug* plug = malloc(sizeof(*plug));

  //printf("%s(): added plug: %p\n", __func__, plug);
  plug->type    = PLUGIN_TYPE_VST;
  plug->shm_num = shm_slot;
#if 0
  snprintf(plug->name, 63, shared->name);
#endif

  pluglist = g_list_append(pluglist, plug);
}
#endif


#if 0
gboolean
plugin_name_valid(const char* plugin_name)
{
  GList* l = pluglist;
  for(;l;l=l->next){
    Plugin* plugin = l->data;
    if(!strcmp(plugin_name, l->data)) return TRUE;
  }
  dbg(0, "plugin name not valid: %s", plugin_name);
  return FALSE;
}
#endif


GdkPixbuf*
plugin_get_icon(FSTInfo* plugin)
{
  char hash_name[16];
  char icon_name[256];

  sprintf(hash_name, "%.8s", plugin->name);
  gchar* lower = g_ascii_strdown(hash_name, -1);
  replace(lower, " ", "_");

  char* t;
  if((t = g_hash_table_lookup(plugins.map, lower))){
    strncpy(icon_name, t, 255);
  }
  else icon_name[0] = '\0';
  GdkPixbuf* icon = strlen(icon_name) ? g_hash_table_lookup(plugin_icons, icon_name) : NULL;
  if(!icon) icon = g_hash_table_lookup(plugin_icons, "no_plugin");

  g_free(lower);
  return icon;
}


plug*
plugin_get_by_name(const char* name)
{
  GList* l = pluglist;
  for(;l;l=l->next){
    Plugin* plugin = l->data;
    if(!strcmp(name, plugin->finfo->name)) return l->data;
  }
  if(strcmp(name, "Bypass") && strcmp(name, NO_PLUGIN) && strcmp(name, "Edit") && strcmp(name, "Separator")) dbg(0, "plugin name not found: %s", name);
  return NULL;
}


static GtkTreeModel*
plugins_create_model()
{
  GtkTreeStore* treestore = plugin_treestore = gtk_tree_store_new(NUM_COLS, GDK_TYPE_PIXBUF, G_TYPE_STRING, G_TYPE_STRING, G_TYPE_UINT, G_TYPE_UINT, G_TYPE_UINT, G_TYPE_STRING);

  return GTK_TREE_MODEL(treestore);
}


static void
print_icon_list(gpointer key, gpointer val, gpointer user_data)
{
  dbg (0, "key=%s val=%p", (char*)key, val);
}


void
plugin_model_update()
{
	PF;
	GtkTreeIter iter;

	gtk_tree_store_clear (plugin_treestore);

	if(debug>1) g_hash_table_foreach(plugin_icons, print_icon_list, NULL);

	GList* l = pluglist;
	for(;l;l=l->next){
		Plugin* plugin = l->data;
		FSTInfo* finfo = plugin->finfo;
		dbg(2, "appending: %s", finfo->name);

		GdkPixbuf* icon = plugin_get_icon(finfo);

		gtk_tree_store_append (plugin_treestore, &iter, NULL);
		gtk_tree_store_set (plugin_treestore, &iter,
                        COL_ICON, icon,
                        COL_NAME, finfo->name,
                        COL_CATEGORY, "vst",
                        COL_N_IN, finfo->numInputs,
                        COL_N_OUT, finfo->numOutputs,
                        COL_DIR, plugin->dir,
                        -1);
	}
}


GtkWidget*
window__new()
{
  plugin_window* window = g_new0(struct _plugin_window, 1);

  window->treestore = plugin_treestore;

  GtkWidget* gtkwindow = gtk_window_new (GTK_WINDOW_TOPLEVEL);
  gtk_window_set_title (GTK_WINDOW(gtkwindow), "Easyfst");

  window->vbox = gtk_vbox_new(NON_HOMOGENOUS, 0);
  //gtk_widget_show(window->vbox);

  //GtkWidget* toolbar = sm_toolbar_new(sm_window, NULL);
  //window__pack(toolbar, FALSE);

  GtkWidget* scrollwin = gtk_scrolled_window_new(NULL, NULL); //adjustments created automatically.
  gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(scrollwin), GTK_POLICY_AUTOMATIC, GTK_POLICY_AUTOMATIC);
  gtk_widget_show(scrollwin);

  //----------------------------------------------
 
  //wrap the model to make the view independently sortable:
  /*GtkTreeModel* */sort_model = gtk_tree_model_sort_new_with_model(GTK_TREE_MODEL(plugin_treestore));

  //GtkWidget* treeview = gtk_tree_view_new_with_model(GTK_TREE_MODEL(plugin_treestore));
  GtkWidget* treeview = gtk_tree_view_new_with_model(GTK_TREE_MODEL(sort_model));
  gtk_widget_show(treeview);
  gtk_container_add(GTK_CONTAINER(scrollwin), treeview);

  //column 0 - icon:
  GtkCellRenderer* cell0 = gtk_cell_renderer_pixbuf_new();
  GtkTreeViewColumn* col0 = gtk_tree_view_column_new_with_attributes("", cell0, "pixbuf", COL_ICON, NULL);
  gtk_tree_view_append_column(GTK_TREE_VIEW(treeview), col0); //pack treeview-column into the treeview
  //gtk_tree_view_column_set_resizable(col0, TRUE);
  gtk_tree_view_column_set_fixed_width(col0, 16);

  //gdk_pixbuf_new_from_file(IMAGES_DIR "ni_logo-16.png", NULL);

  //column 1:
  GtkTreeViewColumn* col1 = gtk_tree_view_column_new();
  gtk_tree_view_column_set_title(col1, "Plugin");
  gtk_tree_view_append_column(GTK_TREE_VIEW(treeview), col1); //pack treeview-column into the treeview
  GtkCellRenderer* renderer1 = gtk_cell_renderer_text_new();
  gtk_tree_view_column_set_resizable(col1, TRUE);
  gtk_tree_view_column_pack_start(col1, renderer1, TRUE);     //pack cellrenderer into treeview-column
  gtk_tree_view_column_set_sort_column_id(col1, COL_NAME);

  gtk_tree_view_column_add_attribute(col1, renderer1, "text", COL_NAME);

  //column2:
  GtkTreeViewColumn* col2 = gtk_tree_view_column_new();
  gtk_tree_view_column_set_title(col2, "Category");
  gtk_tree_view_append_column(GTK_TREE_VIEW(treeview), col2);
  GtkCellRenderer* renderer2 = gtk_cell_renderer_text_new();
  gtk_tree_view_column_pack_start(col2, renderer2, TRUE);
  gtk_tree_view_column_set_resizable(col2, TRUE);
  //g_object_set(G_OBJECT(renderer2), "xalign", 0.0, NULL);

  gtk_tree_view_column_add_attribute(col2, renderer2, "text", COL_CATEGORY);

  //num inputs
  GtkTreeViewColumn* col3 = gtk_tree_view_column_new();
  gtk_tree_view_column_set_title(col3, "In");
  gtk_tree_view_append_column(GTK_TREE_VIEW(treeview), col3);
  GtkCellRenderer* renderer3 = gtk_cell_renderer_text_new();
  gtk_tree_view_column_pack_start(col3, renderer3, TRUE);
  gtk_tree_view_column_add_attribute(col3, renderer3, "text", COL_N_IN);

  //num outputs
  GtkTreeViewColumn* col4 = gtk_tree_view_column_new();
  gtk_tree_view_column_set_title(col4, "Out");
  gtk_tree_view_append_column(GTK_TREE_VIEW(treeview), col4);
  GtkCellRenderer* renderer4 = gtk_cell_renderer_text_new();
  gtk_tree_view_column_pack_start(col4, renderer4, TRUE);
  gtk_tree_view_column_add_attribute(col4, renderer4, "text", COL_N_OUT);

  gtk_tree_sortable_set_sort_column_id(GTK_TREE_SORTABLE(sort_model), COL_NAME, GTK_SORT_ASCENDING);

  //----------------------------------------------

#if 0 
  //set up treeview as a dnd source.
  gtk_drag_source_set(treeview, GDK_BUTTON1_MASK | GDK_BUTTON2_MASK, sm_dnd_file_drag_types, dnd_file_drag_types_count, GDK_ACTION_COPY | GDK_ACTION_MOVE | GDK_ACTION_LINK);
  g_signal_connect (G_OBJECT(treeview), "drag-data-get", G_CALLBACK(plugin_window_drag_dataget), NULL);
#endif

  //----------------------------------------------

  g_signal_connect(treeview, "button-press-event", (GCallback)plugin_treeview_on_button_press, NULL);
  g_signal_connect(treeview, "popup-menu", (GCallback)on_popupmenu, NULL); //triggered by SHIFT-F10
  g_signal_connect(GTK_OBJECT(gtkwindow), "delete_event", G_CALLBACK(destroy_handler), NULL);


  gtk_box_pack_start(GTK_BOX(window->vbox), scrollwin, TRUE, TRUE, 0);

  statusbar = gtk_statusbar_new();
  gtk_statusbar_set_has_resize_grip(GTK_STATUSBAR(statusbar), FALSE);
  //gint context_id = gtk_statusbar_get_context_id(GTK_STATUSBAR(statusbar), "Statusbar example");
  gtk_box_pack_end(GTK_BOX(window->vbox), statusbar, FALSE, FALSE, 0);

  context_menu_init((GtkTreeView*)treeview);

  plugin_model_update();

  gtk_container_add(GTK_CONTAINER(gtkwindow), window->vbox);
  gtk_widget_set_size_request(gtkwindow, 350, 400);
  gtk_widget_show_all(gtkwindow);

  return gtkwindow;
}


void
plugin_window_free()
{
}


static void
plugins_load_icons()
{
  plugin_icons = g_hash_table_new(g_str_hash, g_str_equal);
  plugins.map  = g_hash_table_new(g_str_hash, g_str_equal);

  g_hash_table_insert(plugins.map, "2016_ste", "sr2016");
  g_hash_table_insert(plugins.map, "b4",       "ni");
  g_hash_table_insert(plugins.map, "b4fx",     "ni");
  g_hash_table_insert(plugins.map, "bandpass", "grm");
  g_hash_table_insert(plugins.map, "bassline", "audiorealism");
  g_hash_table_insert(plugins.map, "beast",    "refx");
  g_hash_table_insert(plugins.map, "blue_tube","nomad_factory");
  g_hash_table_insert(plugins.map, "cube",     "virsyn");
  g_hash_table_insert(plugins.map, "redoptor", "d16");
  g_hash_table_insert(plugins.map, "dfx_geom", "dfx");
  g_hash_table_insert(plugins.map, "dfx_tran", "dfx");
  g_hash_table_insert(plugins.map, "crystal",  "green_oak");
  g_hash_table_insert(plugins.map, "izotope_", "izotope");
  g_hash_table_insert(plugins.map, "kontakt2", "ni");
  g_hash_table_insert(plugins.map, "kore",     "ni");
  g_hash_table_insert(plugins.map, "kore fx",  "ni");
  g_hash_table_insert(plugins.map, "northpol", "prosoniq");
  g_hash_table_insert(plugins.map, "ohmboyz_", "ohmboyz");
  g_hash_table_insert(plugins.map, "pianoteq", "pianoteq");
  g_hash_table_insert(plugins.map, "pitchacc", "grm");
  g_hash_table_insert(plugins.map, "psp_vint", "psp");
  g_hash_table_insert(plugins.map, "quadrasi", "refx");
  g_hash_table_insert(plugins.map, "red_wing", "sound_magic");
  g_hash_table_insert(plugins.map, "realisti", "syntheway");
  g_hash_table_insert(plugins.map, "studio_c", "nomad_factory");
  g_hash_table_insert(plugins.map, "tc nativ", "tc");
  g_hash_table_insert(plugins.map, "trilogy",  "ni");
  g_hash_table_insert(plugins.map, "saffirea", "focusrite");
  g_hash_table_insert(plugins.map, "saffirec", "focusrite");
  g_hash_table_insert(plugins.map, "saffiree", "focusrite");
  g_hash_table_insert(plugins.map, "saffirer", "focusrite");
  g_hash_table_insert(plugins.map, "sir_086",  "sir");
  g_hash_table_insert(plugins.map, "spectra",  "kjaerhus");
  g_hash_table_insert(plugins.map, "urs_1970", "urs");
  g_hash_table_insert(plugins.map, "urs_1975", "urs");
  g_hash_table_insert(plugins.map, "urs_1980", "urs");

  GList* files = NULL;
  GList* dirs  = NULL;
  gchar* cwd = g_get_current_dir();
  char* icons_dir = g_build_path(cwd, IMAGES_DIR, NULL);
  g_free(cwd);
  if(!g_file_test(icons_dir, G_FILE_TEST_EXISTS)){
    g_free(icons_dir);
    icons_dir = g_build_path("/", "/usr/share/easyfst", "pics", NULL);
    if(!g_file_test(icons_dir, G_FILE_TEST_EXISTS)){
      gwarn("dir doesnt exist: %s", icons_dir);
    }
  }
  filelist_read(icons_dir, &files, &dirs);

  GList* l = files;
  for(;l;l=l->next){
    char* path = l->data;
    gchar* basename = g_path_get_basename(path);
    char* stripped = malloc(256); // stored in the hashtable. Never free'd.
    filename_remove_extension(basename, stripped);

    GdkPixbuf* pixbuf = gdk_pixbuf_new_from_file(path, NULL);

    if(debug) printf("  icon=%s pixbuf=%p\n", stripped, pixbuf);
    g_hash_table_insert(plugin_icons, stripped, pixbuf);

    g_free(path);
    g_free(basename);
  }
  g_list_free(files);
  g_list_free(dirs);
  g_free(icons_dir);
}


static void
plugin_window_drag_dataget(GtkWidget* widget, GdkDragContext* drag_context, GtkSelectionData* data, guint info, guint time, gpointer window)
{
  //for _outgoing_ drags.
  PF;

#if 0
  char text[256], row_str[256]="";
  //gchar* fname = NULL;
  uint32_t plugin_idx = 0;
  GtkTreeIter iter;

  ASSERT_POINTER(window, "user_data");
  plugin_window* plugin_win = (plugin_window*)window;

  //so which rows are selected?:
  GtkTreeModel* model = (GtkTreeModel*)plugin_win->treestore;
  //GtkTreeSelection* selection = gtk_tree_view_get_selection(GTK_TREE_VIEW(plugin_win->treeview));
  GtkTreeSelection* selection = gtk_tree_view_get_selection(GTK_TREE_VIEW(widget));
  GList* selected_rows = gtk_tree_selection_get_selected_rows(selection, &(model));

  //get the data associated with each of the selected rows:
  GList* row = selected_rows;
  for(; row; row=row->next){
    GtkTreePath* treepath_selection = row->data;
    gtk_tree_model_get_iter(model, &iter, treepath_selection);
    gtk_tree_model_get(model, &iter, COL_IDX, &plugin_idx, -1);
    snprintf(row_str, 255, "plugin=%u", plugin_idx + 1);
  }

  //free the selection list data:     
  g_list_foreach(selected_rows, (gpointer)gtk_tree_path_free, NULL);
  g_list_free(selected_rows);


  //FIXME how do we send multiple items? just append them separated by newlines probably.
  snprintf(text, 255, "plugin:%s%c%c", row_str, 13, 10);
  //if(fname) g_free(fname);

  if(!strlen(row_str)){
    //return a dock string instead of a plugin
    char win_id[STRLEN_WIN_ID];
    window__get_id_string(window, win_id);
    snprintf(text, 255, "dock:%s%c%c", win_id, 13, 10);
  }

  gtk_selection_data_set(data,
                        GDK_SELECTION_TYPE_STRING,
                        8,
                        (unsigned char*)text, strlen(text)
                        );
#endif
}


static gboolean
click_timer_reset()
{
  //mouse double-click timer finished. No double-click.
  click_timer = 0;
  return FALSE;
}


static void
on_double_click(GtkTreePath* path)
{
	GtkTreeModel* model = sort_model;
	GtkTreeIter iter;
	gtk_tree_model_get_iter(sort_model, &iter, path);
	gchar* plugin_name, *plugin_dir;
	gtk_tree_model_get(model, &iter, COL_NAME, &plugin_name, COL_DIR, &plugin_dir, -1);

	dbg (0, "plugin=%s", plugin_name);

	gchar* plugin_path = g_build_filename(plugin_dir, plugin_name, NULL);
	//start_plugin(jvst, plugin_name, plugin_dir);
	start_plugin(jvst, plugin_path);
	g_free(plugin_path);
}


static gboolean
plugin_treeview_on_button_press(GtkWidget* treeview, GdkEventButton* event, gpointer userdata)
{
  if(event->type == GDK_BUTTON_PRESS  &&  event->button == 1){

    if(click_timer){ //double click!
      GtkTreePath* path;
      if(gtk_tree_view_get_path_at_pos(GTK_TREE_VIEW(treeview), (gint) event->x, (gint) event->y, &path, NULL, NULL, NULL)){
        on_double_click(path);
        gtk_tree_path_free(path);
      }
      else{
        dbg(0, "no path! FIXME start dnd?");
        return HANDLED;
      }

      return HANDLED;
    }
    else click_timer = g_timeout_add(200, (gpointer)click_timer_reset, NULL);
  }

  if(event->type == GDK_BUTTON_PRESS  &&  event->button == 3){

    //select row if no row is selected or only one other row is selected

    GtkTreeSelection* selection = gtk_tree_view_get_selection(GTK_TREE_VIEW(treeview));

    if(gtk_tree_selection_count_selected_rows(selection) <= 1){

      //get tree path for row that was clicked:
      GtkTreePath* path;
      if(gtk_tree_view_get_path_at_pos(GTK_TREE_VIEW(treeview), (gint) event->x, (gint) event->y, &path, NULL, NULL, NULL)){
        gtk_tree_selection_unselect_all(selection);
        gtk_tree_selection_select_path(selection, path);
        gtk_tree_path_free(path);
      }
    }

    show_popupmenu(treeview, event, userdata);

    return HANDLED;
  }

  return NOT_HANDLED;
}


static gboolean
on_popupmenu(GtkWidget* treeview, gpointer userdata)
{
  show_popupmenu(treeview, NULL, userdata);

  return HANDLED;
}


static void
show_popupmenu(GtkWidget* treeview, GdkEventButton* event, gpointer userdata)
{
  PF;

  //note: event can be NULL here when called from view_onPopupMenu;
  //      gdk_event_get_time() accepts a NULL argument
  gtk_menu_popup(GTK_MENU(menu),
                   NULL, NULL, NULL, NULL,
                   (event != NULL) ? event->button : 0,
                   gdk_event_get_time((GdkEvent*)event));
}


static GtkWidget*
context_menu_init(GtkTreeView* treeview)
{
  g_return_if_fail(treeview);

  menu = gtk_menu_new();

  GtkWidget* item = gtk_image_menu_item_new_with_label("Run plugin");
  gtk_menu_shell_append(GTK_MENU_SHELL(menu), item);
  g_signal_connect(G_OBJECT(item), "activate", G_CALLBACK(run_selected_plugin), treeview);

  gtk_widget_show_all(menu);

  return menu;
}


static void
run_selected_plugin(GtkWidget* widget, GtkTreeView* treeview)
{
  PF;

  GtkTreeIter iter;
  GtkTreeModel* model = gtk_tree_view_get_model(treeview);

  //what is selected?:
  GtkTreeSelection* selection = gtk_tree_view_get_selection(treeview);
  GList* selectionlist = gtk_tree_selection_get_selected_rows(selection, &(model));
  g_return_if_fail(selectionlist);
  dbg (0, "%i rows selected.", g_list_length(selectionlist));

  int i;
  GList* l = selectionlist;
  for(;l;l=l->next){
    GtkTreePath* treepath_selection = l->data;
    gtk_tree_model_get_iter(model, &iter, treepath_selection);
    gchar* plugin_name;
    gtk_tree_model_get(model, &iter, COL_NAME, &plugin_name, -1);

    dbg (0, "plugin=%s", plugin_name);
    start_plugin(jvst, plugin_name);
  }
  g_list_free(selectionlist);
}


void
statusbar_print(const char* fmt, ...)
{
	va_list ap;
	char text[256];
	if (fmt == NULL)
		*text = '\0';
	else {
		va_start(ap, fmt);
		vsprintf(text, fmt, ap);
		va_end(ap);
	}

	gint cid = gtk_statusbar_get_context_id(GTK_STATUSBAR(statusbar), "Song");
	gtk_statusbar_push(GTK_STATUSBAR(statusbar), cid, text);
}


int
vst_discover_from_path(char* _vstpath)
{
	printf("%s(): scanning: %s\n", __func__, _vstpath);

	void scan_dir(gchar* path)
	{
		GError** error = NULL;
		G_CONST_RETURN gchar *file;
		char filepath[256];
		GDir* dir;
		if((dir = g_dir_open(path, 0, error))){
			while((file = g_dir_read_name(dir))){
				snprintf(filepath, 255, "%s/%s", path, file);
				if(g_file_test(filepath, G_FILE_TEST_IS_DIR)){
					scan_dir(filepath);
				}else{
					gchar* b = g_strrstr(filepath, ".dll");
					if(b == filepath + strlen(filepath) - 4){
						vst_discover(filepath, (char*)file);
					}
				}
			}
			g_dir_close(dir);
		}
	}

	gchar* path = NULL;
	gchar* vstpath = g_strdup(_vstpath);
	while((path = strtok(vstpath, ":")) != 0){
		scan_dir(path);
		vstpath = NULL; //!!
	}
	g_free(vstpath);

	dbg(0, "%i plugins found.", g_list_length(pluglist));

	return 0;
}


int
vst_discover(char* path, char* file)
{
	char up_line[] = "\x1b[A";
	//printf("%s%s\n", up_line, path);
	printf("%s\n", path);

	//note that fst_get_info() does use the cached '.fst' info file if available
	FSTInfo* finfo;
	if (!(finfo = fst_get_info(path))) {
		g_warning("Cannot get VST information from ", path);
		return -1;
	}

	if (!finfo->canProcessReplacing) {
		g_warning("VST plugin %s does not support processReplacing", finfo->name);
	}

	Plugin* plugin = g_new(Plugin, 1);
	plugin->finfo = finfo;
	plugin->dir = g_path_get_dirname(path);

	pluglist = g_list_append(pluglist, plugin);

	return 0;
}


static gboolean
destroy_handler (GtkWidget* widget, GdkEventAny* ev, gpointer user_data)
{
	//JackVST* jvst = (JackVST*)user_data;
	//if(jvst && jvst->fst) fst_destroy_editor (jvst->fst);
	easyfst_quit();
	
	return FALSE;
}

