/*
    Copyright (C) 2004 Paul Davis <paul@linuxaudiosystems.com> 
                       Torben Hohn <torbenh@informatik.uni-bremen.de>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

#include <sys/types.h>
#include <string.h>
#include <stdio.h>
#include <fst.h>
#include <glib.h>
#include <vst/aeffectx.h>

#include <lash/lash.h>
#include "jackvst.h"
#include <windows.h>
#include "ayyi_utils.h"

/* audiomaster.c */

extern long jack_host_callback (AEffect*, long, long, long, void*, float);

/* gtk.c */

extern void gui_init (int* argc, char** argv[]);
//extern int  manage_vst_plugin (JackVST*);
extern void vst_window_new(JackVST*);

/* vsti.c */

extern snd_seq_t * create_sequencer (const char* client_name, int isinput);
extern void *midireceiver(void *arg);

JackVST*       jvst;
lash_client_t* lash_client;
gboolean       lash_is_enabled = true;
int            resume_not_rt = 0;
gboolean       quit = FALSE;

gboolean        start_plugin(JackVST*, const char* plug);
static int      instantiate (JackVST*, const char*);
static gboolean idle_cb     (JackVST*);

void
jack_thread_init (void* data)
{
	//fst_adopt_thread ();
}

int process_callback( jack_nframes_t nframes, void* data) 
{
	int i, o;
	JackVST* jvst = (JackVST*) data;
	AEffect* plugin = jvst->fst->plugin;

	if( !jvst->resume_called ) {
	    jvst->resume_called = TRUE;
	    plugin->dispatcher (plugin, effMainsChanged, 0, 1, NULL, 0.0f);
	}
	for (i = 0; i < plugin->numInputs; ++i) {
		jvst->ins[i]  = (float *) jack_port_get_buffer (jvst->inports[i], nframes);
	}

	for (i = 0; i < plugin->numOutputs; ++i) {
		jvst->outs[i]  = (float *) jack_port_get_buffer (jvst->outports[i], nframes);
	}

	if (jvst->bypassed) {

		if (plugin->numInputs) {
			for (o = 0, i = 0; o < plugin->numOutputs; ++o) {
				memcpy (jvst->outs[o], jvst->ins[i], sizeof (float) * nframes);
				
				if (i < plugin->numOutputs - 1) {
					++i;
				}
			}
		} 
		
	} else if (jvst->muted) {

		for (o = 0, i = 0; o < plugin->numOutputs; ++o) {
			if (jvst->outs[o]) {
				memset (jvst->outs[o], 0, sizeof (float) * nframes);
			}
		}
		
	} else {

		if (jvst->event_queue) {

			jack_ringbuffer_data_t vec[2];
			size_t n;
			size_t i;

			jack_ringbuffer_get_read_vector (jvst->event_queue, vec);

			if ((jvst->events->numEvents = (vec[0].len / sizeof (struct VstMidiEvent))) > 0) {
				jvst->events->reserved  = 0;
				
				for (i = 0, n = 0; n < jvst->events->numEvents; ++n, ++i) {
					jvst->events->events[i] = (struct VstEvent*) (vec[0].buf + (n * sizeof (struct VstMidiEvent)));
				}

				plugin->dispatcher (plugin, effProcessEvents, 0, 0, jvst->events, 0.0f);
			}

			if (vec[1].len && (jvst->events->numEvents = (vec[1].len / sizeof (struct VstMidiEvent)) > 0)) {
				jvst->events->reserved  = 0;

				/* don't initialize i again, continue stuffing events after
				   the last batch.
				 */
				
				for (n = 0; n < jvst->events->numEvents; ++n, ++i) {
					jvst->events->events[i] = (struct VstEvent*) (vec[0].buf + (n * sizeof (struct VstMidiEvent)));
				}

				plugin->dispatcher (plugin, effProcessEvents, 0, 0, jvst->events, 0.0f);
			}
			
			jack_ringbuffer_read_advance (jvst->event_queue, (vec[0].len + vec[1].len));
		}


		if (plugin->flags & effFlagsCanReplacing) {
			
			for (i = 0; i < plugin->numOutputs; ++i) {
				memset (jvst->outs[i], 0, sizeof (float) * nframes);
			}
			plugin->processReplacing (plugin, jvst->ins, jvst->outs, nframes);
			
		} else {
			
			for (i = 0; i < plugin->numOutputs; ++i) {
				memset (jvst->outs[i], 0, sizeof (float) * nframes);
			}
			plugin->process (plugin, jvst->ins, jvst->outs, nframes);
		}
	}

	return 0;      
}

int 
main( int argc, char **argv ) 
{
	char* period;
	char *plug;

	jvst = (JackVST*) calloc (1, sizeof (JackVST));

#ifdef HAVE_LASH

	if ((period = strrchr (argv[0], '.')) != NULL) {
		*period = '\0';
	}
	if ((period = strrchr (argv[0], '.')) != NULL) {
		*period = '\0';
	}
	jvst->lash_args = lash_extract_args(&argc, &argv);
#endif

#if 0
	if (argc < 2) {
		fprintf (stderr, "usage: %s <plugin>\n", argv[0]);
		return 1;
	}
#endif

	//setpgrp();

	gui_init (&argc, &argv);

	jvst->with_editor = 1;

	int i; for (i = 1; i < argc; ++i) {
		if (argv[i][0] == '-') {
			if (argv[i][1] == 'n') {
				jvst->with_editor = 0;
			}
			if (argv[i][1] == 'r') {
				resume_not_rt = 1;
			}
			if (argv[i][1] == 'h') {
				fprintf (stderr, "usage: %s [options]\n    options:\n       -l disable lash.\n", argv[0]);
				return EXIT_SUCCESS;
			}
			if (argv[i][1] == 'l') {
				lash_is_enabled = false;
			}
		} else {
			plug = argv[i];
			break;
		}
	}
	dbg(1, "args done.");

	if (fst_init (NULL)) {
		return 1;
	}

	//--------------------------------------------

	plugins_init();

	window__new();

	//--------------------------------------------

	{
	dbg(0, "attempting to start jack...");

	gchar* argv[7];
	gchar jack_args[][32] = {"/usr/bin/jackd", "-d", "alsa", "-P"};
	int a; for(a=0;a<G_N_ELEMENTS(jack_args);a++) argv[a] = jack_args[a];
	argv[a] = NULL;

	GError* error = NULL;
#if 0
	spawn_sync - cant use cos daemonisation doesnt release it.
	//gint exit_status;
	if(!g_spawn_sync(NULL, argv, NULL, G_SPAWN_SEARCH_PATH,
										NULL, //GSpawnChildSetupFunc child_setup,
										NULL, //gpointer user_data,
										NULL, //gchar **standard_output,
										NULL, //gchar **standard_error,
										NULL, //&exit_status,
										&error
                   )){
		dbg(0, "*** error starting jack.");
	}
#endif
	GPid jack_pid;
	if(!g_spawn_async(NULL, argv, NULL, G_SPAWN_SEARCH_PATH,
										NULL, //GSpawnChildSetupFunc child_setup,
										NULL, //gpointer user_data,
										&jack_pid,
										&error
                   )){
		dbg(0, "*** error starting jack.");
	} else dbg(0, "jack pid: %i", jack_pid);
	}

	//--------------------------------------------

	if (jvst->with_editor) {
		gtk_main ();
	} else {
		while(1) sleep (10);
	}
	
	jack_deactivate(jvst->client);
	return EXIT_SUCCESS;
}

gboolean
start_plugin(JackVST* jvst, const char* plug)
{
	if(instantiate(jvst, plug)) return FALSE;

	vst_window_new(jvst);
	fst_show_editor(jvst->fst);

	g_timeout_add(500, (GSourceFunc)idle_cb, jvst);
	return FALSE;
}

void
easyfst_error(const char* fmt, ...)
{
	va_list ap;
	char text[256];
	if (fmt == NULL)
		*text = '\0';
	else {
		va_start(ap, fmt);
		vsprintf(text, fmt, ap);
		va_end(ap);
	}

	fst_error (text);
	statusbar_print (text);
}

static int
instantiate(JackVST* jvst, const char* plug)
{
	//return non-zero on error.

	PF;

	//char* plugpath = g_build_filename("/usr/share/vst", plug, NULL);

	char* period;
	char* client_name = g_path_get_basename(strdup(plug));
	if ((period = strrchr (client_name, '.')) != NULL) {
		*period = '\0';
	}

	if ((jvst->handle = fst_load (plug)) == NULL) {
		easyfst_error ("can't load plugin %s", plug);
		return 1;
	}

	dbg(0, "calling jack_client_open ...");
	if ((jvst->client = jack_client_open (client_name, JackNullOption, NULL )) == 0) {
		easyfst_error ("can't connect to JACK");
		return 1;
	}

	if ((jvst->fst = fst_instantiate (jvst->handle, jack_host_callback, jvst)) == NULL) {
		easyfst_error ("can't instantiate plugin %s", jvst->fst->handle->name);
		return 1;
	}

	AEffect* plugin = jvst->fst->plugin;
	
	/* set rate and blocksize */

	//printf( "sample_rate... \n" );

	plugin->dispatcher (plugin, effSetSampleRate, 0, 0, NULL, 
			    (float) jack_get_sample_rate (jvst->client));
	plugin->dispatcher (plugin, effSetBlockSize, 0, 
			    jack_get_buffer_size (jvst->client), NULL, 0.0f);

	if( resume_not_rt ) {
	    jvst->resume_called = TRUE;
	    plugin->dispatcher (plugin, effMainsChanged, 0, 1, NULL, 0.0f);
	}
	
	// ok.... plugin is running... lets bind to lash...
	
#ifdef HAVE_LASH
	int flags = LASH_Config_Data_Set;

	lash_event_t *event;
	if(lash_is_enabled){
		lash_client =
			lash_init(jvst->lash_args, jack_get_client_name(jvst->client), flags, LASH_PROTOCOL(2, 0));

		if (!lash_client) {
			fprintf(stderr, "%s: could not initialise lash\n", __FUNCTION__);
			fprintf(stderr, "%s: running fst without lash session-support\n", __FUNCTION__);
			fprintf(stderr, "%s: to enable lash session-support launch the lash server prior fst\n", __FUNCTION__);
			//exit(1);
		}

		if (lash_enabled(lash_client)) {
			event = lash_event_new_with_type(LASH_Client_Name);
			lash_event_set_string(event, jack_get_client_name(jvst->client));
			lash_send_event(lash_client, event);
		}
	}
#endif


	/* set program to zero */
	/* i comment this out because it breaks dfx Geometer
	 * looks like we cant set programs for it
	 *
	plugin->dispatcher (plugin, effSetProgram, 0, 0, NULL, 0.0f); */

	if (jvst->with_editor) {
		if (fst_run_editor (jvst->fst)) {
			easyfst_error ("cannot create editor");
			return 1;
		}
	}

	int vst_version = plugin->dispatcher (plugin, effGetVstVersion, 0, 0, NULL, 0.0f);

	if (vst_version >= 2) {
		
                /* should we send it VST events (i.e. MIDI) */
		
		if ((plugin->flags & effFlagsIsSynth) ||
		    (plugin->dispatcher (plugin, effCanDo, 0, 0,(void*) "receiveVstEvents", 0.0f) > 0)) {

			if ((jvst->seq = create_sequencer (jvst->handle->name, TRUE)) == NULL) {
				easyfst_error ("no sequencer for synth");
			} else {
				jvst->events = (struct VstEvents*) malloc (sizeof (struct VstEvents) - (2*sizeof (struct VstEvent*)) +
									   (1024 * sizeof (struct VstMidiEvent*)));

				jvst->event_queue = jack_ringbuffer_create (1024 * sizeof (struct VstMidiEvent));
				pthread_create (&jvst->midi_thread, NULL, &midireceiver, jvst);
			}
#ifdef HAVE_LASH
			if(lash_is_enabled){
				if( lash_enabled( lash_client ) )
					lash_alsa_client_id(lash_client, (unsigned char)snd_seq_client_id(jvst->seq));
			}
#endif
		}
	}

	jvst->inports = (jack_port_t **) malloc (sizeof(jack_port_t*) * plugin->numInputs);
	jvst->ins = (float **) malloc (sizeof (float *) * plugin->numInputs);
	
	int i; for (i = 0; i < plugin->numInputs; ++i) {
		char buf[64];
		snprintf (buf, sizeof(buf), "in%d", i+1);
		jvst->inports[i] = jack_port_register (jvst->client, buf, JACK_DEFAULT_AUDIO_TYPE, JackPortIsInput, 0);
	}
	
	jvst->outports = (jack_port_t **) malloc (sizeof(jack_port_t*) * plugin->numOutputs);
	jvst->outs = (float **) malloc (sizeof (float *) * plugin->numOutputs);
	
	for (i = 0; i < plugin->numOutputs; ++i) {
		char buf[64];
		snprintf (buf, sizeof(buf), "out%d", i+1);
		jvst->outports[i] = jack_port_register (jvst->client, buf, JACK_DEFAULT_AUDIO_TYPE, JackPortIsOutput, 0);
	}

	//jack_set_thread_init_callback (jvst->client, (JackThreadInitCallback) jack_thread_init, jvst);
	
	jack_set_process_callback (jvst->client, (JackProcessCallback) process_callback, jvst); 
	jack_activate (jvst->client);

#ifdef HAVE_LASH
	if(lash_is_enabled){
		if( lash_enabled( lash_client ) ) {
			event = lash_event_new_with_type(LASH_Jack_Client_Name);
			lash_event_set_string(event, client_name);
			lash_send_event(lash_client, event);
		}
	}
#endif

	return 0;
}


static gboolean
idle_cb(JackVST *jvst)
{
	if (quit) {
		fst_destroy_editor( jvst->fst);
		gtk_main_quit();
		return FALSE;
	}

	if(lash_is_enabled){
	if (lash_enabled(lash_client)) {
	    lash_event_t *event;
	    lash_config_t *config;

	    while ((event = lash_get_event(lash_client))) {
		switch (lash_event_get_type(event)) {
		    case LASH_Quit:
			quit = 1;
			lash_event_destroy(event);
			break;
		    case LASH_Restore_Data_Set:
			printf( "lash_restore... \n" );
			lash_send_event(lash_client, event);
			break;
		    case LASH_Save_Data_Set:
			printf( "lash_save... \n" );
			save_data( jvst );
			lash_send_event(lash_client, event);
			break;
		    case LASH_Server_Lost:
			return 1;
		    default:
			printf("%s: receieved unknown LASH event of type %d",
				__FUNCTION__, lash_event_get_type(event));
			lash_event_destroy(event);
			break;
		}
	    }

	    while ((config = lash_get_config(lash_client))) {
		restore_data(config, jvst);
		lash_config_destroy(config);
	    }
		}
	}

	return TRUE;
}

void
easyfst_quit()
{
	gtk_main_quit();
}
